import { useState } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import numerologyService, { CalculateNumerologyRequest, SaveCalculationRequest } from '../services/numerology';
import { CalculationResult, SavedCalculation, RootInterpretation } from '../services/api';
import { useAuth } from './useAuth';

export function useNumerology() {
  const { isAuthenticated } = useAuth();
  const queryClient = useQueryClient();
  const [currentCalculation, setCurrentCalculation] = useState<CalculationResult | null>(null);

  // Calculate numerology
  const calculateMutation = useMutation({
    mutationFn: (data: CalculateNumerologyRequest) => numerologyService.calculateNumerology(data),
    onSuccess: (result) => {
      setCurrentCalculation(result);
    },
  });

  // Save calculation
  const saveMutation = useMutation({
    mutationFn: (data: SaveCalculationRequest) => numerologyService.saveCalculation(data),
    onSuccess: () => {
      // Invalidate user calculations query to refresh the list
      queryClient.invalidateQueries({ queryKey: ['userCalculations'] });
    },
  });

  // Delete calculation
  const deleteMutation = useMutation({
    mutationFn: (id: number) => numerologyService.deleteCalculation(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['userCalculations'] });
    },
  });

  // Get user calculations (only if authenticated)
  const userCalculationsQuery = useQuery({
    queryKey: ['userCalculations'],
    queryFn: () => numerologyService.getUserCalculations(),
    enabled: isAuthenticated,
  });

  // Get root interpretation
  const getInterpretation = (rootNumber: number) =>
    useQuery({
      queryKey: ['interpretation', rootNumber],
      queryFn: () => numerologyService.getInterpretation(rootNumber),
      enabled: rootNumber >= 1 && rootNumber <= 9,
    });

  const calculate = async (data: CalculateNumerologyRequest) => {
    try {
      const result = await calculateMutation.mutateAsync(data);
      return { success: true, data: result };
    } catch (error: any) {
      return { 
        success: false, 
        error: error.message || 'Calculation failed' 
      };
    }
  };

  const saveCalculation = async (data?: SaveCalculationRequest) => {
    if (!currentCalculation) {
      return { success: false, error: 'No calculation to save' };
    }

    try {
      const saveData = data || {
        calculation: currentCalculation.calculation,
        swot_combinations: currentCalculation.swot_combinations,
        interpretation: currentCalculation.interpretation,
        business_guidance: currentCalculation.business_guidance,
        learning_style: currentCalculation.learning_style,
      };

      const result = await saveMutation.mutateAsync(saveData);
      return { success: true, data: result };
    } catch (error: any) {
      return { 
        success: false, 
        error: error.message || 'Failed to save calculation' 
      };
    }
  };

  const deleteCalculation = async (id: number) => {
    try {
      await deleteMutation.mutateAsync(id);
      return { success: true };
    } catch (error: any) {
      return { 
        success: false, 
        error: error.message || 'Failed to delete calculation' 
      };
    }
  };

  return {
    currentCalculation,
    setCurrentCalculation,
    calculate,
    saveCalculation,
    deleteCalculation,
    getInterpretation,
    userCalculations: userCalculationsQuery.data || [],
    isCalculating: calculateMutation.isPending,
    isSaving: saveMutation.isPending,
    isDeleting: deleteMutation.isPending,
    isLoadingUserCalculations: userCalculationsQuery.isLoading,
    calculationError: calculateMutation.error?.message,
    saveError: saveMutation.error?.message,
    deleteError: deleteMutation.error?.message,
    userCalculationsError: userCalculationsQuery.error?.message,
  };
}