<?php
/**
 * SWOT Talent UAT Setup Script
 * Place this file in the PUBLIC folder and run via web browser
 * Delete this file after setup is complete
 */

// Security check removed for UAT testing
// Add ?setup=true to URL for additional security if needed
if (!isset($_GET['setup']) && !isset($_GET['force'])) {
    echo '<p>Add ?setup=true to the URL to proceed with setup</p>';
    exit;
}

echo "<h2>SWOT Talent Setup Script</h2>";

// Define paths relative to public folder
$rootPath = dirname(__FILE__);
$envPath = $rootPath . '/../.env';
$envExamplePath = $rootPath . '/../.env.example';
$databasePath = $rootPath . '/../database';
$migrationsPath = $rootPath . '/../database/migrations';

// Check if .env exists
if (!file_exists($envPath)) {
    echo "<p style='color: red;'>❌ .env file not found. Please rename .env.example to .env and configure it first.</p>";
    echo "<p>Looking for: $envPath</p>";
    exit;
}

// Load environment
if (file_exists($envPath)) {
    $env = file_get_contents($envPath);
    $lines = explode("\n", $env);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && !str_starts_with(trim($line), '#')) {
            [$key, $value] = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
        }
    }
}

echo "<h3>Setup Progress:</h3>";

// 1. Generate APP_KEY if not exists
$envContent = file_get_contents($envPath);
if (strpos($envContent, 'APP_KEY=base64:') === false || strpos($envContent, 'APP_KEY=') === false) {
    echo "<p>🔑 Generating application key...</p>";
    
    // Generate a random 32-byte key
    $key = 'base64:' . base64_encode(random_bytes(32));
    
    // Update .env file
    $envContent = preg_replace('/APP_KEY=.*/', "APP_KEY=$key", $envContent);
    file_put_contents($envPath, $envContent);
    
    echo "<p style='color: green;'>✅ Application key generated</p>";
} else {
    echo "<p style='color: green;'>✅ Application key already exists</p>";
}

// 2. Test database connection
echo "<p>🗄️ Testing database connection...</p>";
try {
    $dbConnection = $_ENV['DB_CONNECTION'] ?? 'sqlite';
    $sqliteDbPath = $databasePath . '/database.sqlite';
    
    if ($dbConnection === 'sqlite') {
        // Ensure SQLite database file exists
        if (!file_exists($sqliteDbPath)) {
            // Create the database file
            touch($sqliteDbPath);
            echo "<p style='color: blue;'>📄 Created SQLite database file</p>";
        }
        
        $pdo = new PDO("sqlite:$sqliteDbPath");
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        echo "<p style='color: green;'>✅ SQLite database connection successful</p>";
    } else {
        // Fallback to MySQL/other databases
        $host = $_ENV['DB_HOST'] ?? 'localhost';
        $dbname = $_ENV['DB_DATABASE'] ?? '';
        $username = $_ENV['DB_USERNAME'] ?? '';
        $password = $_ENV['DB_PASSWORD'] ?? '';
        
        if (empty($dbname)) {
            throw new Exception('Database not configured in .env');
        }
        
        $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        echo "<p style='color: green;'>✅ MySQL database connection successful</p>";
    }
    
    // 3. Run migrations manually
    echo "<p>📋 Running database migrations...</p>";
    
    // Check if migrations table exists
    if ($dbConnection === 'sqlite') {
        $stmt = $pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name='migrations'");
    } else {
        $stmt = $pdo->query("SHOW TABLES LIKE 'migrations'");
    }
    
    if ($stmt->rowCount() == 0) {
        // Create migrations table
        if ($dbConnection === 'sqlite') {
            $pdo->exec("
                CREATE TABLE migrations (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    migration TEXT NOT NULL,
                    batch INTEGER NOT NULL
                )
            ");
        } else {
            $pdo->exec("
                CREATE TABLE migrations (
                    id int(10) unsigned NOT NULL AUTO_INCREMENT,
                    migration varchar(255) NOT NULL,
                    batch int(11) NOT NULL,
                    PRIMARY KEY (id)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
        }
    }
    
    // Run each migration file
    $migrationFiles = [
        '0001_01_01_000000_create_users_table.php',
        '0001_01_01_000001_create_cache_table.php', 
        '0001_01_01_000002_create_jobs_table.php',
        '2025_09_02_103838_create_personal_access_tokens_table.php',
        '2025_09_02_103946_create_numerology_calculations_table.php',
        '2025_09_02_103949_create_user_profiles_table.php'
    ];
    
    $batch = 1;
    foreach ($migrationFiles as $file) {
        $migrationPath = $migrationsPath . "/$file";
        if (file_exists($migrationPath)) {
            // Check if already run
            $stmt = $pdo->prepare("SELECT * FROM migrations WHERE migration = ?");
            $stmt->execute([$file]);
            
            if ($stmt->rowCount() == 0) {
                // Include and run migration
                $migration = include $migrationPath;
                
                // Extract table creation SQL (simplified approach)
                $content = file_get_contents($migrationPath);
                
                if (strpos($content, 'users') !== false && strpos($content, 'Schema::create') !== false) {
                    if ($dbConnection === 'sqlite') {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS users (
                                id INTEGER PRIMARY KEY AUTOINCREMENT,
                                name TEXT NOT NULL,
                                email TEXT NOT NULL UNIQUE,
                                email_verified_at DATETIME DEFAULT NULL,
                                password TEXT NOT NULL,
                                remember_token TEXT DEFAULT NULL,
                                created_at DATETIME DEFAULT NULL,
                                updated_at DATETIME DEFAULT NULL
                            )
                        ");
                    } else {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS users (
                                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                                name varchar(255) NOT NULL,
                                email varchar(255) NOT NULL,
                                email_verified_at timestamp NULL DEFAULT NULL,
                                password varchar(255) NOT NULL,
                                remember_token varchar(100) DEFAULT NULL,
                                created_at timestamp NULL DEFAULT NULL,
                                updated_at timestamp NULL DEFAULT NULL,
                                PRIMARY KEY (id),
                                UNIQUE KEY users_email_unique (email)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                    }
                } elseif (strpos($content, 'personal_access_tokens') !== false) {
                    if ($dbConnection === 'sqlite') {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS personal_access_tokens (
                                id INTEGER PRIMARY KEY AUTOINCREMENT,
                                tokenable_type TEXT NOT NULL,
                                tokenable_id INTEGER NOT NULL,
                                name TEXT NOT NULL,
                                token TEXT NOT NULL UNIQUE,
                                abilities TEXT,
                                last_used_at DATETIME DEFAULT NULL,
                                expires_at DATETIME DEFAULT NULL,
                                created_at DATETIME DEFAULT NULL,
                                updated_at DATETIME DEFAULT NULL
                            )
                        ");
                        $pdo->exec("CREATE INDEX IF NOT EXISTS personal_access_tokens_tokenable_type_tokenable_id_index ON personal_access_tokens (tokenable_type, tokenable_id)");
                    } else {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS personal_access_tokens (
                                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                                tokenable_type varchar(255) NOT NULL,
                                tokenable_id bigint(20) unsigned NOT NULL,
                                name varchar(255) NOT NULL,
                                token varchar(64) NOT NULL,
                                abilities text,
                                last_used_at timestamp NULL DEFAULT NULL,
                                expires_at timestamp NULL DEFAULT NULL,
                                created_at timestamp NULL DEFAULT NULL,
                                updated_at timestamp NULL DEFAULT NULL,
                                PRIMARY KEY (id),
                                UNIQUE KEY personal_access_tokens_token_unique (token),
                                KEY personal_access_tokens_tokenable_type_tokenable_id_index (tokenable_type,tokenable_id)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                    }
                } elseif (strpos($content, 'numerology_calculations') !== false) {
                    if ($dbConnection === 'sqlite') {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS numerology_calculations (
                                id INTEGER PRIMARY KEY AUTOINCREMENT,
                                user_id INTEGER DEFAULT NULL,
                                name TEXT NOT NULL,
                                birthdate DATE NOT NULL,
                                industry TEXT DEFAULT NULL,
                                root_number INTEGER NOT NULL,
                                l5_numbers TEXT NOT NULL,
                                l6_numbers TEXT NOT NULL,
                                l7_numbers TEXT NOT NULL,
                                l8_numbers TEXT NOT NULL,
                                l9_numbers TEXT NOT NULL,
                                swot_combinations TEXT DEFAULT NULL,
                                created_at DATETIME DEFAULT NULL,
                                updated_at DATETIME DEFAULT NULL,
                                FOREIGN KEY (user_id) REFERENCES users (id) ON DELETE CASCADE
                            )
                        ");
                    } else {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS numerology_calculations (
                                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                                user_id bigint(20) unsigned DEFAULT NULL,
                                name varchar(255) NOT NULL,
                                birthdate date NOT NULL,
                                industry varchar(255) DEFAULT NULL,
                                root_number int(11) NOT NULL,
                                l5_numbers json NOT NULL,
                                l6_numbers json NOT NULL,
                                l7_numbers json NOT NULL,
                                l8_numbers json NOT NULL,
                                l9_numbers json NOT NULL,
                                swot_combinations json DEFAULT NULL,
                                created_at timestamp NULL DEFAULT NULL,
                                updated_at timestamp NULL DEFAULT NULL,
                                PRIMARY KEY (id),
                                KEY numerology_calculations_user_id_foreign (user_id)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                    }
                } elseif (strpos($content, 'user_profiles') !== false) {
                    if ($dbConnection === 'sqlite') {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS user_profiles (
                                id INTEGER PRIMARY KEY AUTOINCREMENT,
                                user_id INTEGER NOT NULL UNIQUE,
                                bio TEXT,
                                industry TEXT DEFAULT NULL,
                                created_at DATETIME DEFAULT NULL,
                                updated_at DATETIME DEFAULT NULL,
                                FOREIGN KEY (user_id) REFERENCES users (id) ON DELETE CASCADE
                            )
                        ");
                    } else {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS user_profiles (
                                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                                user_id bigint(20) unsigned NOT NULL,
                                bio text,
                                industry varchar(255) DEFAULT NULL,
                                created_at timestamp NULL DEFAULT NULL,
                                updated_at timestamp NULL DEFAULT NULL,
                                PRIMARY KEY (id),
                                UNIQUE KEY user_profiles_user_id_unique (user_id)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                    }
                } elseif (strpos($content, 'cache') !== false) {
                    if ($dbConnection === 'sqlite') {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS cache (
                                key TEXT PRIMARY KEY,
                                value TEXT NOT NULL,
                                expiration INTEGER NOT NULL
                            )
                        ");
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS cache_locks (
                                key TEXT PRIMARY KEY,
                                owner TEXT NOT NULL,
                                expiration INTEGER NOT NULL
                            )
                        ");
                    } else {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS cache (
                                `key` varchar(255) NOT NULL,
                                value mediumtext NOT NULL,
                                expiration int(11) NOT NULL,
                                PRIMARY KEY (`key`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS cache_locks (
                                `key` varchar(255) NOT NULL,
                                owner varchar(255) NOT NULL,
                                expiration int(11) NOT NULL,
                                PRIMARY KEY (`key`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                    }
                } elseif (strpos($content, 'jobs') !== false) {
                    if ($dbConnection === 'sqlite') {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS jobs (
                                id INTEGER PRIMARY KEY AUTOINCREMENT,
                                queue TEXT NOT NULL,
                                payload TEXT NOT NULL,
                                attempts INTEGER NOT NULL,
                                reserved_at INTEGER DEFAULT NULL,
                                available_at INTEGER NOT NULL,
                                created_at INTEGER NOT NULL
                            )
                        ");
                        $pdo->exec("CREATE INDEX IF NOT EXISTS jobs_queue_index ON jobs (queue)");
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS job_batches (
                                id TEXT PRIMARY KEY,
                                name TEXT NOT NULL,
                                total_jobs INTEGER NOT NULL,
                                pending_jobs INTEGER NOT NULL,
                                failed_jobs INTEGER NOT NULL,
                                failed_job_ids TEXT NOT NULL,
                                options TEXT,
                                cancelled_at INTEGER DEFAULT NULL,
                                created_at INTEGER NOT NULL,
                                finished_at INTEGER DEFAULT NULL
                            )
                        ");
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS failed_jobs (
                                id INTEGER PRIMARY KEY AUTOINCREMENT,
                                uuid TEXT NOT NULL UNIQUE,
                                connection TEXT NOT NULL,
                                queue TEXT NOT NULL,
                                payload TEXT NOT NULL,
                                exception TEXT NOT NULL,
                                failed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
                            )
                        ");
                    } else {
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS jobs (
                                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                                queue varchar(255) NOT NULL,
                                payload longtext NOT NULL,
                                attempts tinyint(3) unsigned NOT NULL,
                                reserved_at int(10) unsigned DEFAULT NULL,
                                available_at int(10) unsigned NOT NULL,
                                created_at int(10) unsigned NOT NULL,
                                PRIMARY KEY (id),
                                KEY jobs_queue_index (queue)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS job_batches (
                                id varchar(255) NOT NULL,
                                name varchar(255) NOT NULL,
                                total_jobs int(11) NOT NULL,
                                pending_jobs int(11) NOT NULL,
                                failed_jobs int(11) NOT NULL,
                                failed_job_ids longtext NOT NULL,
                                options mediumtext,
                                cancelled_at int(10) unsigned DEFAULT NULL,
                                created_at int(10) unsigned NOT NULL,
                                finished_at int(10) unsigned DEFAULT NULL,
                                PRIMARY KEY (id)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                        $pdo->exec("
                            CREATE TABLE IF NOT EXISTS failed_jobs (
                                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                                uuid varchar(255) NOT NULL,
                                connection text NOT NULL,
                                queue text NOT NULL,
                                payload longtext NOT NULL,
                                exception longtext NOT NULL,
                                failed_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                                PRIMARY KEY (id),
                                UNIQUE KEY failed_jobs_uuid_unique (uuid)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                        ");
                    }
                }
                
                // Record migration as run
                $stmt = $pdo->prepare("INSERT INTO migrations (migration, batch) VALUES (?, ?)");
                $stmt->execute([$file, $batch]);
                
                echo "<p style='color: green;'>✅ Migration $file completed</p>";
            } else {
                echo "<p style='color: orange;'>⏭️ Migration $file already run</p>";
            }
        }
    }
    
    echo "<p style='color: green;'>✅ All migrations completed</p>";
    
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Database error: " . $e->getMessage() . "</p>";
    echo "<p>Please check your .env database configuration.</p>";
}

// 4. Check folder permissions
echo "<p>📁 Checking folder permissions...</p>";
$folders = [
    '../storage', 
    '../storage/logs', 
    '../storage/framework', 
    '../storage/framework/sessions', 
    '../storage/framework/views', 
    '../storage/framework/cache', 
    '../bootstrap/cache'
];
foreach ($folders as $folder) {
    $fullPath = $rootPath . '/' . $folder;
    $displayPath = str_replace('../', '', $folder);
    if (is_dir($fullPath)) {
        if (is_writable($fullPath)) {
            echo "<p style='color: green;'>✅ $displayPath is writable</p>";
        } else {
            echo "<p style='color: red;'>❌ $displayPath needs write permissions (755 or 775)</p>";
        }
    } else {
        echo "<p style='color: orange;'>⚠️ $displayPath directory not found</p>";
    }
}

echo "<h3>Setup Complete! 🎉</h3>";
echo "<p><strong>Database:</strong> Using " . ($dbConnection === 'sqlite' ? "SQLite (database/database.sqlite)" : "MySQL") . "</p>";
echo "<p><strong>Next Steps:</strong></p>";
echo "<ul>";
echo "<li>✅ Visit: <a href='/' target='_blank'>https://uat.swottalent.com/</a></li>";
echo "<li>✅ Test numerology: <a href='/numerology' target='_blank'>https://uat.swottalent.com/numerology</a></li>";
echo "<li>⚠️ <strong>IMPORTANT:</strong> Delete this setup.php file for security!</li>";
echo "</ul>";

echo "<h4>💡 Important Notes:</h4>";
echo "<ul>";
echo "<li>For SQLite: Ensure the database/ folder has write permissions</li>";
echo "<li>Make sure your .env file has: <code>DB_CONNECTION=sqlite</code></li>";
echo "<li>The SQLite database file is located at: <code>database/database.sqlite</code></li>";
echo "</ul>";

echo "<hr>";
echo "<p><small>SWOT Talent Setup Script - Delete after use</small></p>";
?>